package easik.sketch.util.Export;

import java.util.ArrayList;
import java.util.HashMap;


import org.xml.sax.Attributes;
import org.xml.sax.helpers.DefaultHandler;

import easik.sketch.util.Export.Components.ExportEdge;
import easik.sketch.util.Export.Components.ExportPath;
import easik.sketch.util.Export.Components.ExportTable;
import easik.sketch.util.Export.Components.ExportUniqueKey;
import easik.sketch.util.Export.Constraints.ExportCommutativeDiagram;
import easik.sketch.util.Export.Constraints.ExportConstraint;
import easik.sketch.util.Export.Constraints.ExportProductConstraint;
import easik.sketch.util.Export.Constraints.ExportPullback;
import easik.sketch.util.Export.Constraints.ExportSumConstraint;

/**
 * A class to parse XML specifically for database and SQL file generation.
 * Should not be used for any other purpose, as it does not contain any graphical information.
 * 
 * 
 * @author Vera Ranieri
 * @since 2006-05-18 Vera Ranieri
 * @version 2006-06-27 Vera Ranieri
 */
public class ExportHandler extends DefaultHandler{
	
	/**
	 * A HashMap of all the tables of the graph.  Indexed by the name of the table. 
	 */
	private HashMap<String, ExportTable> _tables;
	
	/**
	 * The database platform to export to
	 */
	private String _platform;
	
	/**
	 * The current table being added to the database
	 */
	private ExportTable _newTable;
	/**
	 * The current path
	 */
	private ExportPath _curPath;
	/**
	 * The current constraint
	 */
	private ExportConstraint _curConstraint;
	/**
	 * The current unique key
	 */
	private ExportUniqueKey _curUniqueKey;
	/**
	 * The current datatype
	 */
	private String _curDatatype;
	
	/** 
	 * HashMap of all edges to add to paths.  Indexed by the name of the edge
	 */
	private HashMap<String, ExportEdge> _edges;
	
	/** 
	 * HashMap of all paths in the sketch.  Indexed by the name of the path
	 */
	private HashMap<String, ExportPath> _paths;
	
	/** 
	 * A list of all constraints found in the XML file.
	 */
	private ArrayList<ExportConstraint> _constraints;
	
	/**
	 * A hashmap of all data types found in the XML file.  Indexed by the description of the type
	 */
	private HashMap<String, String> _datatypes;
	
	/**
	 * A count maintained to ensure that each procedure has a unique identifier
	 */
	private int _count;

	/**
	 * Create a new instance of a SQLHandler and  initialize all fields.
	 * @param platform The string representation of the user chosen platform.
	 *
	 *@since 2006-05-18 Vera Ranieri
	 */
	public ExportHandler(String platform){
		_tables = new HashMap<String, ExportTable>();
		_edges = new HashMap<String, ExportEdge>();
		_paths = new HashMap<String, ExportPath>();
		_constraints = new ArrayList<ExportConstraint>();
		_platform = platform;
		_datatypes = new HashMap<String, String>();
		_count = 0;
		
	}
	
	/**
	 * Gets the platform of the database.
	 * 
	 * @return The platform name
	 * @since 2006-05-18, Vera Ranieri
	 */
	public String getPlatform(){
		return _platform;
	}
	
	/**
	 * Getter method for the HashMap of all tables to be created. 
	 * (ie all entities in XML and their properties)
	 * 
	 * @return The tables of the XML
	 * @since 2006-05-18 Vera Ranieri
	 */
	public HashMap<String, ExportTable> getTables(){
		return _tables;
	}
	
	/**
	 * Gets the table associated with a given entity name
	 * @param name The name of the table
	 * @return The SQLTable associated with that name
	 * 
	 * @since 2006-05-18, Vera Ranieri
	 */
	public ExportTable getTable(String name){
		return (ExportTable)_tables.get(name);
	}
	
	/**
	 * Gets the constraints associated with a XML sketch
	 * 
	 * @return ArrayList of all constraints
	 * @since 2006-05-23, Vera Ranieri
	 */
	public ArrayList<ExportConstraint> getConstraints(){
		return _constraints;
	}
	
	/**
	 * Overloaded method from the extended Default Handler
	 * @see org.xml.sax.helpers.DefaultHandler
	 * @param namespace @see org.xml.sax.helpers.DefaultHandler
	 * @param localName @see org.xml.sax.helpers.DefaultHandler
	 * @param qName @see org.xml.sax.helpers.DefaultHandler
	 * @param atts @see org.xml.sax.helpers.DefaultHandler
	 * @since 2006-05-18, Vera Ranieri
	 */
	public void startElement(String namespace, String localName, String qName, Attributes atts) {

		if (qName.equals("entity")) {
			String name = atts.getValue("name");
			
			_newTable = new ExportTable(name);
			_tables.put(name, _newTable);
		} 
		else if (qName.equals("attribute")){
			_newTable.addAttribute(atts.getValue("name"), atts.getValue("attributeType"), _datatypes);
		}
		else if (qName.equals("uniqueKey")){
			_curUniqueKey = new ExportUniqueKey();
			_newTable.addUniqueKey(_curUniqueKey);
		}
		else if (qName.equals("attref")){
			_curUniqueKey.addElement(atts.getValue("name"));
		}
		else if (qName.equals("datatype")){
			_curDatatype = atts.getValue("name");
		}
		else if(qName.equals(_platform)){
			_datatypes.put(_curDatatype, atts.getValue("type"));
			
		}
		else if (qName.equals("edge")) {
			String source = atts.getValue("source");
			String target = atts.getValue("target");
			
			boolean injective = atts.getValue("injective").equals("true");
			String name = atts.getValue("id");
			
			ExportEdge e = new ExportEdge(name, injective, source, target);
			
			ExportTable t = (ExportTable)_tables.get(source);
			t.addEdge(e);
			
			_edges.put(name, e);
			
		} 
		else if(qName.equals("path")){
			String name= atts.getValue("id");
			String domain = atts.getValue("domain");
			_curPath = new ExportPath(name, domain);
			_paths.put(name, _curPath);
		}

		else if(qName.equals("edgeref")){
			_curPath.addEdge((ExportEdge)_edges.get(atts.getValue("id")));
		}
		
		else if (qName.equals("sumconstraint") )
		{
			_curConstraint = new ExportSumConstraint(_count);
			_constraints.add(_curConstraint);
		}
		else if ( qName.equals("pullbackconstraint")){
			_curConstraint = new ExportPullback(_count);
			_constraints.add(_curConstraint);
		}
		else if (qName.equals("productconstraint")){
			_curConstraint = new ExportProductConstraint(_count);
			_constraints.add(_curConstraint);
		}
		else if(qName.equals("commutativediagram")){
			_curConstraint = new ExportCommutativeDiagram(_count);
			_constraints.add(_curConstraint);
		}
		else if (qName.equals("pathref")){
			ExportPath tempPath = (ExportPath)_paths.get(atts.getValue("id"));
			_curConstraint.addPath(tempPath);
		}
	}

	/**
	 * Overloaded method from the extended default handler.
	 * @see org.xml.sax.helpers.DefaultHandler
	 * @param uri
	 * @param localName
	 * @param qName
	 * @since 2006-05-23, Vera Ranieri
	 */
	public void endElement(String uri, String localName, String qName) {
		
		if (qName.equals("pullbackconstraint") || qName.equals("productconstraint") 
				|| qName.equals("sumconstraint") ||  qName.equals("commutativediagram")) {
			_curConstraint.setBaseTable();
			_curConstraint.setTargetTable();
			_count++;
		}
	}
	
}

